let pixelsPerCm = 10; // Standard-Kalibrierungsfaktor
let armLengthPx = null;
let strideLengthPx = null;
let bodyHeightPx = null;

async function main() {
  const video = document.getElementById("camera");
  const canvas = document.getElementById("overlay");
  const stickmanCanvas = document.getElementById("stickman");
  const ctx = canvas.getContext("2d");
  const stickmanCtx = stickmanCanvas.getContext("2d");

  const calibrationSlider = document.getElementById("calibrationSlider");
  const calibrationValue = document.getElementById("calibrationValue");
  const measureButton = document.getElementById("measureButton");

  const armLengthOutput = document.getElementById("armLength");
  const strideLengthOutput = document.getElementById("strideLength");
  const bodyHeightOutput = document.getElementById("bodyHeight");

  // Kamera einrichten
  async function setupCamera() {
  try {
    // Kamera-Zugriff anfordern
    const constraints = {
      video: {
        facingMode: "environment" // Rückkamera bevorzugt (alternativ: "user" für Frontkamera)
      }
    };

    const stream = await navigator.mediaDevices.getUserMedia(constraints);

    // Video und Canvas-Elemente verknüpfen
    const video = document.getElementById("camera");
    const canvas = document.getElementById("overlay");
    const stickmanCanvas = document.getElementById("stickman");

    video.srcObject = stream;

    return new Promise((resolve) => {
      video.onloadedmetadata = () => {
        // Kameraauflösung setzen
        canvas.width = video.videoWidth;
        canvas.height = video.videoHeight;
        stickmanCanvas.width = video.videoWidth;
        stickmanCanvas.height = video.videoHeight;

        console.log(`Kameraauflösung: ${video.videoWidth}x${video.videoHeight}`);
        resolve();
      };
    });
  } catch (error) {
    console.error("Kamera konnte nicht verwendet werden:", error);
    alert("Bitte Kamera-Berechtigungen aktivieren.");
  }
}
  await setupCamera();

  // Pose Detection mit MoveNet
  const detector = await poseDetection.createDetector(
    poseDetection.SupportedModels.MoveNet, {
      modelType: poseDetection.movenet.modelType.SINGLEPOSE_LIGHTNING,
      modelUrl: '/production/models/movenet/model.json'
    }
  );

  // Verbindungen zwischen den Punkten (Körpersegmente)
  const connections = [
    ["nose", "left_eye"],
    ["nose", "right_eye"],
    ["left_eye", "left_ear"],
    ["right_eye", "right_ear"],
    ["left_shoulder", "right_shoulder"],
    ["left_shoulder", "left_elbow"],
    ["left_elbow", "left_wrist"],
    ["right_shoulder", "right_elbow"],
    ["right_elbow", "right_wrist"],
    ["left_shoulder", "left_hip"],
    ["right_shoulder", "right_hip"],
    ["left_hip", "right_hip"],
    ["left_hip", "left_knee"],
    ["left_knee", "left_ankle"],
    ["right_hip", "right_knee"],
    ["right_knee", "right_ankle"]
  ];

  // Funktion zur Berechnung der Werte
  function calculatePixelValues(keypoints) {
    const leftShoulder = keypoints.find(k => k.name === 'left_shoulder');
    const leftWrist = keypoints.find(k => k.name === 'left_wrist');
    const rightShoulder = keypoints.find(k => k.name === 'right_shoulder');
    const rightWrist = keypoints.find(k => k.name === 'right_wrist');

    const leftHip = keypoints.find(k => k.name === 'left_hip');
    const leftAnkle = keypoints.find(k => k.name === 'left_ankle');
    const rightHip = keypoints.find(k => k.name === 'right_hip');
    const rightAnkle = keypoints.find(k => k.name === 'right_ankle');

    const nose = keypoints.find(k => k.name === 'nose');

    // Armlänge
    armLengthPx = leftShoulder && leftWrist
      ? Math.sqrt(Math.pow(leftWrist.x - leftShoulder.x, 2) + Math.pow(leftWrist.y - leftShoulder.y, 2))
      : rightShoulder && rightWrist
      ? Math.sqrt(Math.pow(rightWrist.x - rightShoulder.x, 2) + Math.pow(rightWrist.y - rightShoulder.y, 2))
      : null;

    // Schrittlänge
    strideLengthPx = leftHip && leftAnkle
      ? Math.sqrt(Math.pow(leftAnkle.x - leftHip.x, 2) + Math.pow(leftAnkle.y - leftHip.y, 2))
      : rightHip && rightAnkle
      ? Math.sqrt(Math.pow(rightAnkle.x - rightHip.x, 2) + Math.pow(rightAnkle.y - rightHip.y, 2))
      : null;

    // Körpergröße
    bodyHeightPx = nose && (leftAnkle || rightAnkle)
      ? Math.sqrt(Math.pow((leftAnkle || rightAnkle).x - nose.x, 2) + Math.pow((leftAnkle || rightAnkle).y - nose.y, 2))
      : null;
  }

  // Funktion zur Aktualisierung der Ergebnisse
  function updateResults() {
    if (armLengthPx !== null) {
      armLengthOutput.textContent = `Armlänge: ${(armLengthPx / pixelsPerCm).toFixed(2)} cm`;
    }
    if (strideLengthPx !== null) {
      strideLengthOutput.textContent = `Schrittlänge: ${(strideLengthPx / pixelsPerCm).toFixed(2)} cm`;
    }
    if (bodyHeightPx !== null) {
      bodyHeightOutput.textContent = `Körpergröße: ${(bodyHeightPx / pixelsPerCm).toFixed(2)} cm`;
    }
  }

  // Funktion zum Zeichnen des Strichmännchens
  function drawStickman(keypoints) {
    stickmanCtx.clearRect(0, 0, stickmanCanvas.width, stickmanCanvas.height);
    stickmanCtx.strokeStyle = "blue";
    stickmanCtx.lineWidth = 2;

    connections.forEach(([start, end]) => {
      const startPoint = keypoints.find(k => k.name === start && k.score > 0.5);
      const endPoint = keypoints.find(k => k.name === end && k.score > 0.5);

      if (startPoint && endPoint) {
        stickmanCtx.beginPath();
        stickmanCtx.moveTo(startPoint.x, startPoint.y);
        stickmanCtx.lineTo(endPoint.x, endPoint.y);
        stickmanCtx.stroke();
      }
    });
  }

  // Pose-Erkennung starten
  async function detectPose() {
    const poses = await detector.estimatePoses(video);

    ctx.clearRect(0, 0, canvas.width, canvas.height);
    if (poses.length > 0) {
      const keypoints = poses[0].keypoints;

      // Punkte und Verbindungen auf dem Overlay zeichnen
      keypoints.forEach(({ x, y, score }) => {
        if (score > 0.5) {
          ctx.beginPath();
          ctx.arc(x, y, 5, 0, 2 * Math.PI);
          ctx.fillStyle = "green";
          ctx.fill();
        }
      });

      connections.forEach(([start, end]) => {
        const startPoint = keypoints.find(k => k.name === start && k.score > 0.5);
        const endPoint = keypoints.find(k => k.name === end && k.score > 0.5);

        if (startPoint && endPoint) {
          ctx.beginPath();
          ctx.moveTo(startPoint.x, startPoint.y);
          ctx.lineTo(endPoint.x, endPoint.y);
          ctx.strokeStyle = "green";
          ctx.lineWidth = 2;
          ctx.stroke();
        }
      });

      // Strichmännchen zeichnen
      drawStickman(keypoints);

      // Bei Klick auf "Messen" Pixel-Werte berechnen
      measureButton.addEventListener("click", () => {
        calculatePixelValues(keypoints);
        updateResults();
      });
    }

    requestAnimationFrame(detectPose);
  }

  detectPose();

  // Kalibrierung anpassen
  calibrationSlider.addEventListener("input", () => {
    pixelsPerCm = parseFloat(calibrationSlider.value);
    calibrationValue.textContent = pixelsPerCm.toFixed(1);
    updateResults();
  });

  async function getCameraResolution() {
  const stream = await navigator.mediaDevices.getUserMedia({ video: true });
  const videoTrack = stream.getVideoTracks()[0]; // Zugriff auf den Video-Track
  const settings = videoTrack.getSettings(); // Einstellungen des Tracks
    console.log(settings);
    
  const width = settings.width;
  const height = settings.height;

  console.log(`Kameraauflösung: ${width}x${height}`);
  
  // Optional: Kameraauflösung im DOM anzeigen
  const resolutionOutput = document.getElementById("resolutionOutput");
  if (resolutionOutput) {
    resolutionOutput.textContent = `Kameraauflösung: ${width}x${height}`;
  }

  // Video-Stream stoppen, falls nicht mehr benötigt
  videoTrack.stop();
}

getCameraResolution();
}

main();



async function getAvailableCameras() {
  try {
    const devices = await navigator.mediaDevices.enumerateDevices();
    const videoDevices = devices.filter(device => device.kind === "videoinput");

    const cameraSelect = document.getElementById("cameraSelect");

    // Kameraauswahl füllen
    videoDevices.forEach((device, index) => {
      const option = document.createElement("option");
      option.value = device.deviceId;
      option.text = device.label || `Kamera ${index + 1}`;
      cameraSelect.appendChild(option);
    });

    cameraSelect.addEventListener("change", async () => {
      const selectedDeviceId = cameraSelect.value;
      await setupCameraWithDeviceId(selectedDeviceId);
    });
  } catch (error) {
    console.error("Kameras konnten nicht aufgelistet werden:", error);
  }
}

async function setupCameraWithDeviceId(deviceId) {
  const constraints = {
    video: {
      deviceId: { exact: deviceId }
    }
  };

  const stream = await navigator.mediaDevices.getUserMedia(constraints);
  const video = document.getElementById("camera");
  video.srcObject = stream;
}
